
# Print Report: Chimera
# Copyright 2005 by Alexander V. Christensen

""" Print the current report. """

#    This file is part of GanttPV.
#
#    GanttPV is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    GanttPV is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with GanttPV; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# 050910 - first version; a complete redesign of prior printing scripts
# 060204 - numerous improvements, including better display of time-scales, and lighter background colors
# 060401 - added more time scales

import wx, sys


# define constants

RepeatHeaders = True
AlwaysFillPage = False
CellBoundary = 1

TextPen = wx.BLACK_PEN
DarkBorder = wx.BLACK_PEN
LightBorder = wx.Pen((160, 160, 160))

PageOrder = "LRTB"
  # if "LRTB", pages are printed left-right, then top-bottom
  # if "TBLR", pages are printed top-bottom, then left-right

PortraitSwap = (sys.platform == 'darwin')
  # if True, when printing portrait, swap x and y dimensions
  # should be true for MacOS, but false for Windows

MonthNames = ['January', 'February', 'March', 'April', 'May', 'June',
    'July', 'August', 'September', 'October', 'November', 'December']

ShortMonthNames = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
    'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']

ColorReplacement = {
    Data.Option.get('DeletedColor'): wx.Colour(255, 204, 204),
    Data.Option.get('ParentColor'): wx.Colour(204, 204, 255),
    Data.Option.get('HiddenColor'): wx.Colour(238, 238, 238),
    Data.Option.get('NotWorkDay'): wx.Colour(238, 238, 238),
    Data.Option.get('WorkDay'): wx.WHITE,
    Data.Option.get('ChildColor'): wx.WHITE,
    }


# get report information

ReportID = self.ReportID
Report = Data.Report[ReportID]
ProjectID = Report.get('ProjectID')
Project = Data.Project[ProjectID]
ReportName = Report.get('Name') + ' (' + Project.get('Name') + ')'

if isinstance(self.Report, wx.grid.Grid):
    ReportStyle = "Grid"
    grid = self.Report

    RowCount = grid.GetNumberRows()
    RowHeights = [grid.GetRowSize(r) for r in xrange(RowCount)]
    RowLabelSize = grid.GetRowLabelSize()

    ColumnCount = grid.GetNumberCols()
    ColumnWidths = [grid.GetColSize(c) for c in xrange(ColumnCount)]
    ColumnLabelSize = grid.GetColLabelSize()

    Columns = grid.table.columns
    ColumnTypes = grid.table.ctypes
    ColumnOffsets = grid.table.coloffset

    font = grid.GetLabelFont()
    FontSize = font.GetPointSize()
    LabelFont = wx.Font(FontSize, wx.SWISS, wx.NORMAL, wx.BOLD)

elif isinstance(self.Report, wx.ListCtrl):
    ReportStyle = "List"
    list = self.Report
    ImageList = list.GetImageList(wx.IMAGE_LIST_SMALL)

    RowCount = list.GetItemCount()
    RowHeights = [22] * RowCount
    RowLabelSize = 40

    ColumnCount = list.GetColumnCount()
    ColumnWidths = [list.GetColumnWidth(c) for c in xrange(ColumnCount)]
    ColumnLabelSize = 32

    Columns = list.columns
    ColumnTypes = list.ctypes
    ColumnOffsets = list.coloffset

    font = wx.SystemSettings_GetFont(wx.SYS_DEFAULT_GUI_FONT)
    FontSize = font.GetPointSize()
    TextFont = wx.Font(FontSize, wx.SWISS, wx.NORMAL, wx.NORMAL)
    LabelFont = wx.Font(FontSize, wx.SWISS, wx.NORMAL, wx.BOLD)

else:
    if debug: print "unknown report style"
    raise NotImplementedError

TitleSize = FontSize * 2
RowHeaderSize = RowLabelSize + CellBoundary * 2
ColumnHeaderSize = ColumnLabelSize + CellBoundary * 2
PageHeaderSize = ColumnHeaderSize + TitleSize

x = RowHeaderSize
CumWidths = [x]
for width in ColumnWidths:
    x += width + CellBoundary
    CumWidths.append(x)
CumWidths.append(CellBoundary)

y = PageHeaderSize
CumHeights = [y]
for height in RowHeights:
    y += height + CellBoundary
    CumHeights.append(y)
CumHeights.append(CellBoundary)


# functions to divide the report into pages

def GetColumnCount():
    if ReportStyle == "Grid":
        return grid.GetNumberCols()
    else: # ReportStyle == "List"
        return list.GetColumnCount()

def GetRowCount():
    if ReportStyle == "Grid":
        return grid.GetNumberRows()
    else: # ReportStyle == "List"
        return list.GetItemCount()

def GetPages(page, header, cells):
    """ Divide a list of column widths (or row heights) into pages

    page -- the width (or height) of each page
    header -- the width (or height) of the header
    cells -- a list of column widths (or row heights)

    Returns a two-tuple:
      - list of page boundaries.  Each boundary is a tuple of the column
        (or row) index and the pixel offset into that column (or row).
      - true, if okay to repeat headers on each page; false, otherwise
    """

    page = max(int(page), 1)

    Pages = [(-1, x) for x in xrange(0, header, page)]
    cutoff = -header % page  # unused space on current page

    repeat = RepeatHeaders and (header < page / 2)
    if repeat:
        page -= header

    for index, size in enumerate(cells):
        size += CellBoundary
        if size > cutoff:
            if AlwaysFillPage or (size % page <= cutoff):
                X = cutoff
            else:
                X = 0
            Pages += [(index, x) for x in xrange(X, size, page)]
            cutoff = (X - size) % page
        else:
            cutoff -= size

    Pages.append((len(cells), 0))
    return Pages, repeat

def GetPartialWidth(col_start, col_end):
    return CumWidths[col_end] - CumWidths[col_start] + CellBoundary

def GetPartialHeight(row_start, row_end):
    return CumHeights[row_end] - CumHeights[row_start] + CellBoundary


# time scale functions

def GetTimeScaleHeader(c):
    name1 = name2 = ''

    colid = Columns[c]
    of = ColumnOffsets[c]
    date = Data.GetColumnDate(colid, of)
    if date == None: return

    try:
        year, month, day = Data._date_to_ymd(date)
        WeekSize = Data.WeekSize
    except AttributeError:
        date_str = Data.DateIndex[date]
        parts = date_str.split('-')
        year, month, day = [int(p) for p in parts]
        WeekSize = 7

    ctid = ColumnTypes[c]
    if ctid not in Data.ColumnType:
        return
    ct = Data.ColumnType[ctid]
    ctperiod, ctfield = ct['Name'].split('/', 1)
    period = ct.get('PeriodSize') or ctperiod
    if '*' in period:
        type, size = period.split('*', 1)
        size = int(size)
    else:
        type, size = period, 1

    if type == 'Day':
        if of == 0 or day <= size:
            name1 = MonthNames[month-1] + ' ' + str(year)
        name2 = str(day)
    elif type == 'Week':
        if of == 0 or day <= size * WeekSize:
            name1 = ShortMonthNames[month-1] + ' ' + str(year)[-2:]
        name2 = str(day)
    elif type == 'Month':
        if of == 0 or month <= size:
            name1 = str(year)
        name2 = str(month)
    elif type == 'Quarter':
        if of == 0 or month <= size * 3:
            name1 = str(year)
        name2 = 'Q' + str((month + 2) / 3)
    elif type == 'Year':
        if of == 0 or (year % 100) < size:
            name1 = str(year)
        name2 = str(year)[-2:]

    return name1, name2


# drawing helper functions

def ScaleDC(dc, size):
    w, h = dc.GetSize().Get()
    # if debug: print "dc size:", (w, h)
    scaleX = float(w) / size.width
    scaleY = float(h) / size.height
    scale = min(scaleX, scaleY)
    dc.SetUserScale(scale, scale)
    # if debug: print "scale:", scale

def ClipText(dc, text, size):
    """ Returns a portion of text no larger than size. """
    if '\n' in text:
        result = []
        y = 0
        for line in text.splitlines():
            y += dc.GetTextExtent(line)[1]
            if y > size.height:
                break
            line = ClipText(dc, line, size)
            result.append(line)
        return '\n'.join(result)

    width, height = dc.GetTextExtent(text)
    if height > size.height:
        return ""
    if width > size.width:
        max_width = size.width - dc.GetTextExtent(" ...")[0]
        extents = dc.GetPartialTextExtents(text)
        for index, x in enumerate(extents):
            if x > max_width:
                return text[:index] + " ..."

    return text

def DrawClippedText(dc, text, rect):
    size = wx.Size(rect.width - 2, rect.height - 2)
    text = ClipText(dc, text, size)
    dc.DrawLabel(text, rect, wx.ALIGN_CENTER, -1)

def DrawBox(dc, rect):
    if CellBoundary:
        x, y, width, height = rect.Get()
        right = x + width - CellBoundary
        bottom = y + height - CellBoundary
        dc.SetPen(DarkBorder)
        dc.DrawLine(x, y, x, bottom) # left
        dc.DrawLine(x, y, right, y) # top
        dc.DrawLine(right, y, right, bottom) # right
        dc.DrawLine(x, bottom, right, bottom) # bottom
        dc.DrawPoint(right, bottom)


# functions to draw the report

def DrawTitle(dc, left, top):
    dc.SetBackgroundMode(wx.TRANSPARENT)
    dc.SetPen(TextPen)
    dc.SetFont(LabelFont)
    dc.DrawText(ReportName, left, top)

def DrawCorner(dc, left, top):
    rect = wx.Rect(left, top, RowHeaderSize, ColumnHeaderSize)
    DrawBox(dc, rect)

def DrawColumnHeaders(dc, left, top, start, end):
    if start < 0:
        DrawCorner(dc, left, top)
        left += RowHeaderSize - CellBoundary
        start = 0

    if start >= end:
        return

    x = right = left + CellBoundary
    y = top + CellBoundary
    bottom = y + ColumnLabelSize

    half = ColumnLabelSize / 2  # used for time-scale headers
    middle = y + half
    upper_half = wx.Rect(left, y, CumWidths[end] - CumWidths[start], half)
    c1 = end

    dc.SetBackgroundMode(wx.TRANSPARENT)
    dc.SetFont(LabelFont)

    for c in xrange(start, end):
        width = ColumnWidths[c]
        right = x + width
        rect = wx.Rect(x, y, width, ColumnLabelSize)

        if ReportStyle == "Grid":
            text = grid.GetColLabelValue(c)
        else: # ReportStyle == "List"
            text = list.GetColumn(c).GetText()

        dc.SetPen(TextPen)

        of = ColumnOffsets[c]
        if of < 0:
            DrawClippedText(dc, text, rect)
            if CellBoundary:
                x -= CellBoundary
                dc.SetPen(DarkBorder)
                dc.DrawLine(x, y, x, bottom) # left
        else:
            # time-scale headers have two parts
            header = GetTimeScaleHeader(c)
            if header:
                name1, name2 = header
            else:
                lines = text.splitlines()
                name1, name2 = lines[0], lines[-1]

            # search previous pages for an upper header
            if (c == start) and (of > 0) and (not name1):
                for c0 in xrange(c - 1, c - of - 1, -1):
                    header = GetTimeScaleHeader(c0)
                    if not header:
                        break
                    name1 = header[0]
                    if name1:
                        break
                x1 = x - CumWidths[c] + CumWidths[c0]
            else:
                c0 = c
                x1 = x

            # draw upper header
            if name1:
                for c1 in xrange(c + 1, ColumnCount):
                    if ColumnOffsets[c1] < 1:
                        break
                    header = GetTimeScaleHeader(c1)
                    if (not header) or header[0]:
                        break
                else:
                    c1 = ColumnCount

                width1 = CumWidths[c1] - CumWidths[c0]
                rect1 = wx.Rect(x1, y, width1, half)

                dc.SetClippingRect(upper_half)
                DrawClippedText(dc, name1, rect1)
                dc.DestroyClippingRegion()

                if CellBoundary:
                    x -= CellBoundary
                    dc.SetPen(DarkBorder)
                    dc.DrawLine(x, y, x, bottom) # left
            else:
                if CellBoundary:
                    x -= CellBoundary
                    dc.SetPen(DarkBorder)
                    dc.DrawLine(x, middle, x, bottom) # left

            if CellBoundary:
                dc.DrawLine(x, middle, right, middle) # middle

            # draw lower header
            rect2 = wx.Rect(x, middle, width, half)
            DrawClippedText(dc, name2, rect2)

        x = right + CellBoundary
    bottom = y + ColumnLabelSize
    width = x - left

    if CellBoundary:
        dc.SetPen(DarkBorder)
        if c1 > end:
            dc.DrawLine(right, middle, right, bottom) # right
        else:
            dc.DrawLine(right, top, right, bottom) # right
        dc.DrawLine(left, top, right, top) # top
        dc.DrawLine(left, bottom, right, bottom) # bottom
        dc.DrawPoint(right, bottom)

def DrawRowHeaders(dc, left, top, start, end):
    if start < 0:
        DrawCorner(dc, left, top + TitleSize)
        top += PageHeaderSize - CellBoundary
        start = 0

    if start >= end:
        return

    x = left + CellBoundary
    y = bottom = top + CellBoundary
    right = x + RowLabelSize

    dc.SetBackgroundMode(wx.TRANSPARENT)
    dc.SetFont(LabelFont)

    for r in xrange(start, end):
        height = RowHeights[r]
        bottom = y + height
        rect = wx.Rect(x, y, RowLabelSize, height)

        dc.SetPen(TextPen)
        DrawClippedText(dc, str(r+1), rect)

        if CellBoundary:
            y -= CellBoundary
            dc.SetPen(DarkBorder)
            dc.DrawLine(left, y, right, y) # top

        y = bottom + CellBoundary
    right = x + RowLabelSize
    height = y - top

    if CellBoundary:
        dc.SetPen(DarkBorder)
        dc.DrawLine(left, top, left, bottom) # left
        dc.DrawLine(right, top, right, bottom) # right
        dc.DrawLine(left, bottom, right, bottom) # bottom
        dc.DrawPoint(right, bottom)

def DrawCells(dc, left, top, col_start, col_end, row_start, row_end):
    if row_start < 0:
        DrawColumnHeaders(dc, left, top + TitleSize, col_start, col_end)
        top += PageHeaderSize - CellBoundary
        row_start = 0

    if col_start < 0:
        DrawRowHeaders(dc, left, top, row_start, row_end)
        left += RowHeaderSize - CellBoundary
        col_start = 0

    if (row_start >= row_end) or (col_start >= col_end):
        return

    x = right = left + CellBoundary
    y = bottom = top + CellBoundary

    for r in xrange(row_start, row_end):
        x = left + CellBoundary
        height = RowHeights[r]
        bottom = y + height

        for c in xrange(col_start, col_end):
            width = ColumnWidths[c]
            right = x + width
            rect = wx.Rect(x, y, width, height)

            if ReportStyle == "Grid":
                rend = grid.GetCellRenderer(r, c)
                attr = grid.GetOrCreateCellAttr(r, c).Clone()

                bgcolor = grid.GetCellBackgroundColour(r, c)
                bgcolor = ColorReplacement.get(bgcolor.Get(), bgcolor)
                attr.SetBackgroundColour(bgcolor)

                font = grid.GetCellFont(r, c)
                fontSize = font.GetPointSize()
                textFont = wx.Font(fontSize, wx.SWISS, wx.NORMAL, wx.NORMAL)
                attr.SetFont(textFont)

                selected = 0
                # selected = grid.IsInSelection(r, c)
                rend.Draw(grid, attr, dc, rect, r, c, selected)
            else: # ReportStyle == "List"
                text = list.OnGetItemText(r, c)
                img = list.OnGetItemImage(r)
                attr = list.OnGetItemAttr(r)
                bgcolor = attr.GetBackgroundColour()
                bgcolor = ColorReplacement.get(bgcolor.Get(), bgcolor)

                dc.SetPen(wx.TRANSPARENT_PEN)
                dc.SetBrush(wx.Brush(bgcolor))
                dc.SetFont(TextFont)
                dc.DrawRectangle(x, y, width, height)

                rect.Deflate(2, 2)
                if (c > 0) or (img < 0):
                    text = ClipText(dc, text, rect.GetSize())
                    dc.DrawLabel(text, rect, wx.ALIGN_CENTER_VERTICAL, -1)
                else:
                    image = ImageList.GetBitmap(img)
                    size = wx.Size(width - image.GetWidth() - 6, height - 2)
                    text = ClipText(dc, text, size)
                    dc.DrawImageLabel(text, image, rect,
                        wx.ALIGN_CENTER_VERTICAL, -1)

            if CellBoundary:
                x -= CellBoundary
                y -= CellBoundary

                # use lighter boundaries within time scales

                if ColumnOffsets[c] < 1:
                     dc.SetPen(DarkBorder)
                else:
                     dc.SetPen(LightBorder)
                dc.DrawLine(x, y, x, bottom) # left

                if ColumnOffsets[c] < 0 or r == 0:
                     dc.SetPen(DarkBorder)
                     dc.DrawLine(x, y, right, y) # top
                else:
                     dc.SetPen(LightBorder)
                     dc.DrawLine(x + CellBoundary, y, right, y) # top

                y += CellBoundary

            x = right + CellBoundary
        y = bottom + CellBoundary
    width = x - left
    height = y - top

    if CellBoundary:
        dc.SetPen(DarkBorder)
        dc.DrawLine(right, top, right, bottom) # right
        dc.DrawLine(left, bottom, right, bottom) # bottom
        dc.DrawPoint(right, bottom)


def DrawReport(dc):
    dc.BeginDrawing()
    dc.SetBackground(wx.WHITE_BRUSH)
    dc.Clear()
    DrawCells(dc, 0, 0, -1, ColumnCount, -1, RowCount)
    DrawTitle(dc, 0, 0)
    dc.EndDrawing()

class ReportCanvas(wx.ScrolledWindow):

    def __init__(self, parent):
        wx.ScrolledWindow.__init__(self, parent, style=wx.SUNKEN_BORDER)
        self.SetBackgroundColour(wx.WHITE)

        self.width = GetPartialWidth(-1, ColumnCount)
        self.height = GetPartialHeight(-1, RowCount)
        self.bitmap = wx.EmptyBitmap(self.width, self.height)
        dc = wx.BufferedDC(None, self.bitmap)
        DrawReport(dc)

        self.SetScrollRate(20, 20)
        self.Bind(wx.EVT_PAINT, self.OnPaint)
        self.SetScale(1)

    def OnPaint(self, event):
        dc = wx.PaintDC(self)
        self.PrepareDC(dc)
        dc.SetUserScale(self.scale, self.scale)
        dc.DrawBitmap(self.bitmap, 10, 10)

    def SetScale(self, scale):
        width = (self.width + 20) * scale
        height = (self.height + 20) * scale
        self.SetVirtualSize((width, height))
        self.scale = scale
        self.Refresh()


class ReportPrintout(wx.Printout):

    def __init__(self, paperSize, margins, scale):
        wx.Printout.__init__(self, ReportName)

        res = wx.ScreenDC().GetPPI()
        xScale = res.x * scale / 25.0  # screen pixels per page millimeter
        yScale = res.y * scale / 25.0

        width, height = paperSize.Get()
        left, top, right, bottom = margins

        printWidth = (width - left - right) * xScale
        self.ColumnPages, self.RowHeaderRepeat = GetPages(printWidth,
            RowHeaderSize, ColumnWidths)

        printHeight = (height - top - bottom) * yScale
        self.RowPages, self.ColumnHeaderRepeat = GetPages(printHeight,
            PageHeaderSize, RowHeights)

        self.paperSize = wx.Size(width * xScale, height * yScale)
        self.origin = wx.RealPoint(left * xScale, top * yScale)
        self.max = (len(self.ColumnPages) - 1) * (len(self.RowPages) - 1)

    def HasPage(self, page):
        return (1 <= page <= self.max)

    def GetPageInfo(self):
        return (1, self.max, 1, self.max)

    def OnPrintPage(self, page):
        """ Draw a page to the DC """
        if PageOrder == "TBLR":
            colpage, rowpage = divmod(page - 1, len(self.RowPages) - 1)
        else: # assume "LRTB"
            rowpage, colpage = divmod(page - 1, len(self.ColumnPages) - 1)

        # determine columns and rows
        col_start, x_offset = self.ColumnPages[colpage]
        col_end, x_excess = self.ColumnPages[colpage+1]

        row_start, y_offset = self.RowPages[rowpage]
        row_end, y_excess = self.RowPages[rowpage+1]

        col_max = GetColumnCount()
        if col_end > col_max:
            col_end, x_excess = col_max, 0

        row_max = GetRowCount()
        if row_end > row_max:
            row_end, y_excess = row_max, 0

        # determine coordinates
        left, top = self.origin.Get()

        x = left - x_offset
        row_header = (self.RowHeaderRepeat and (col_start >= 0))
        if row_header:
            x += RowHeaderSize - CellBoundary
            titleX = left
        else:
            titleX = x - CumWidths[col_start] + CellBoundary

        y = top - y_offset
        column_header = (self.ColumnHeaderRepeat and (row_start >= 0))
        if column_header:
            y += PageHeaderSize - CellBoundary
            headerY = top + TitleSize
            titleY = top
        else:
            titleY = y  # not used unless row_start < 0

        # adjustments for incomplete rows or columns
        right = x + GetPartialWidth(col_start, col_end) + x_excess
        bottom = y + GetPartialHeight(row_start, row_end) + y_excess

        if x_excess: col_end += 1
        if y_excess: row_end += 1

        # perform the drawing
        dc = self.GetDC()
        dc.BeginDrawing()
        dc.SetBackground(wx.WHITE_BRUSH)
        dc.Clear()
        ScaleDC(dc, self.paperSize)

        DrawCells(dc, x, y, col_start, col_end, row_start, row_end)

        if row_header:
            dc.SetPen(wx.TRANSPARENT_PEN)
            dc.SetBrush(wx.WHITE_BRUSH)
            dc.DrawRectangle(left, y, RowHeaderSize, bottom - top)
            DrawRowHeaders(dc, left, y, row_start, row_end)

        if column_header:
            dc.SetPen(wx.TRANSPARENT_PEN)
            dc.SetBrush(wx.WHITE_BRUSH)
            dc.DrawRectangle(x, top, right - left, PageHeaderSize)
            DrawColumnHeaders(dc, x, headerY, col_start, col_end)

            if row_header:
                dc.SetPen(wx.TRANSPARENT_PEN)
                dc.SetBrush(wx.WHITE_BRUSH)
                dc.DrawRectangle(left, top, RowHeaderSize, PageHeaderSize)
                DrawCorner(dc, left, headerY)

        # clear the margins
        w, h = dc.GetSize().Get()
        xScale, yScale = dc.GetUserScale()
        width = w / xScale + 1
        height = h / yScale + 1

        dc.SetPen(wx.TRANSPARENT_PEN)
        dc.SetBrush(wx.WHITE_BRUSH)
        dc.DrawRectangle(0, 0, left, height) # left
        dc.DrawRectangle(0, 0, width, top) # top
        if x_excess:
            dc.DrawRectangle(width, 0, right - width, height) # right
        if y_excess:
            dc.DrawRectangle(0, height, width, bottom - height) # bottom

        # draw the title
        if column_header or (row_start < 0):
            if not self.RowHeaderRepeat:
                dc.SetClippingRegion(left, titleY, right - left, TitleSize)
            DrawTitle(dc, titleX, titleY)
            dc.DestroyClippingRegion()

        dc.EndDrawing()
        return True


class PreviewFrame(wx.PreviewFrame):
    def __init__(self, preview, parent, title="Print Preview"):
        wx.PreviewFrame.__init__(self, preview, parent, title)
        self.preview = preview

        menus = wx.MenuBar()
        printMenu = wx.Menu()
        printMenu.Append(wx.ID_CLOSE, "Close\tCtrl-W", "Close print preview")
        menus.Append(printMenu, 'Preview')

        helpMenu = wx.Menu()
        helpMenu.Append(wx.ID_ABOUT, "About GanttPV", "")
        helpMenu.Append(ID.HOME_PAGE, "GanttPV Home")
        helpMenu.Append(ID.HELP_PAGE, "Help Page", "URL of Help Text")
        helpMenu.Append(ID.FORUM, "Forum", "URL of Forum")
        menus.Append(helpMenu, "&Help")

        self.SetMenuBar(menus)
        self.Bind(wx.EVT_MENU, self.doClose, id=wx.ID_CLOSE)

    def doClose(self, event):
         self.Close()


class PrintFrame(wx.Frame):

    def __init__(self, *args, **kwds):
        wx.Frame.__init__(self, *args, **kwds)

        self.SetTitle('Print')
        self.SetSize((400, 350))
        self.SetBackgroundColour((238, 238, 238))

        # perform layout
        border_sizer = wx.BoxSizer(wx.VERTICAL)
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        self.canvas = ReportCanvas(self)
        main_sizer.Add(self.canvas, 1, wx.GROW | wx.ALL, 2)

        button_sizer = wx.BoxSizer(wx.HORIZONTAL)

        btn = wx.Button(self, -1, "Page Setup")
        btn.Bind(wx.EVT_BUTTON, self.doPageSetup)
        button_sizer.Add(btn, 1, wx.ALL, 4)

        btn = wx.Button(self, -1, "Preview")
        btn.Bind(wx.EVT_BUTTON, self.doPreview)
        button_sizer.Add(btn, 1, wx.ALL, 4)

        btn = wx.Button(self, -1, "Print...")
        btn.Bind(wx.EVT_BUTTON, self.doPrint)
        button_sizer.Add(btn, 1, wx.ALL, 4)

        main_sizer.Add(button_sizer, 0, wx.GROW | wx.ALL, 2)

        scale_sizer = wx.BoxSizer(wx.HORIZONTAL)
        label =  wx.StaticText(self, -1, 'Scale:')
        scale_sizer.Add(label, 0, wx.CENTER | wx.ALL, 2)

        self.scaler = wx.TextCtrl(self, size=(50, 25), style=wx.TE_PROCESS_ENTER)
        self.scaler.Bind(wx.EVT_TEXT, self.doScale)
        scale_sizer.Add(self.scaler, 0, wx.CENTER | wx.ALL, 2)

        label = wx.StaticText(self, -1, '%')
        scale_sizer.Add(label, 0, wx.CENTER | wx.ALL, 2)
        scale_sizer.Add((20, 20))
        self.message = wx.StaticText(self)
        scale_sizer.Add(self.message, 0, wx.CENTER | wx.ALL, 2)
        main_sizer.Add(scale_sizer, 0, wx.GROW | wx.ALL, 2)

        border_sizer.Add(main_sizer, 1, wx.GROW | wx.ALL, 10)
        self.SetSizer(border_sizer)
        self.CentreOnScreen()
        self.Bind(wx.EVT_SIZE, self.onSize)

        # create menus
        menus = wx.MenuBar()

        fileMenu = wx.Menu()
        fileMenu.Append(wx.ID_CLOSE, "Close\tCtrl-W", "Close window")
        fileMenu.Append(wx.ID_EXIT, "Quit\tCtrl-Q", "Quit program")
        menus.Append(fileMenu, "File")

        helpMenu = wx.Menu()
        helpMenu.Append(wx.ID_ABOUT, "About GanttPV")
        helpMenu.Append(ID.HOME_PAGE, "GanttPV Home")
        helpMenu.Append(ID.HELP_PAGE, "Help Page", "URL of Help Text")
        helpMenu.Append(ID.FORUM, "Forum", "URL of Forum")
        menus.Append(helpMenu, "&Help")

        self.SetMenuBar(menus)
        self.Bind(wx.EVT_MENU, self.doClose, id=wx.ID_CLOSE)
        self.Bind(wx.EVT_CLOSE, self.onClose)

        # initialize settings
        self._read_data()

    def _read_data(self):
        self.printdata = wx.PrintData()

        orientation = Report.get('PageOrientation') or wx.PORTRAIT
        if orientation == 'portrait':
            orientation = wx.PORTRAIT
        elif orientation == 'landscape':
            orientation = wx.LANDSCAPE
        self.printdata.SetOrientation(orientation)

        paperId = Report.get('PaperId') or wx.PAPER_LETTER
        self.printdata.SetPaperId(paperId)

        width = Report.get('PaperWidth') or 216
        height = Report.get('PaperHeight') or 279
        self.printdata.SetPaperSize((width, height))

        self.pagedata = wx.PageSetupDialogData(self.printdata)

        left = Report.get('MarginLeft') or 6
        top = Report.get('MarginTop') or 6
        self.pagedata.SetMarginTopLeft((left, top))

        right = Report.get('MarginRight') or 6
        bottom = Report.get('MarginBottom') or 14
        self.pagedata.SetMarginBottomRight((right, bottom))

        self.pdd = wx.PrintDialogData(self.printdata)

        scale = Report.get('PrintScale') or 100
        self.scale = scale
        self.scaler.SetValue(str(scale))

        self.changed = False

    def _save_data(self):
        if not self.changed: return

        change = { 'Table': 'Report', 'ID': ReportID }
        change['PrintScale'] = self.scale

        change['PageOrientation'] = self.printdata.GetOrientation()
        change['PaperId'] = self.printdata.GetPaperId()

        size = self.printdata.GetPaperSize()
        change['PaperWidth'] = size.width
        change['PaperHeight'] = size.height

        topleft = self.pagedata.GetMarginTopLeft()
        change['MarginLeft'] = topleft.x
        change['MarginTop'] = topleft.y

        bottomright = self.pagedata.GetMarginBottomRight()
        change['MarginRight'] = bottomright.x
        change['MarginBottom'] = bottomright.y

        Data.Update(change)
        Data.SetUndo('Edit Print Settings')

    def _apply_scale(self, scale):
        if scale < 20:
            scale = 20
            self.message.SetLabel('minimum scale is 20%')
        elif scale > 500:
            scale = 500
            self.message.SetLabel('maximum scale is 500%')
        else:
            self.message.SetLabel('')
        self.scale = scale
        # self.canvas.SetScale(scale / 100.0)

    def _read_margins(self):
        topLeft = self.pagedata.GetMarginTopLeft()
        bottomRight = self.pagedata.GetMarginBottomRight()
        size = self.printdata.GetPaperSize()

        portrait = (self.printdata.GetOrientation() == wx.PORTRAIT)
        if portrait and PortraitSwap:
            top, left = topLeft.Get()
            bottom, right = bottomRight.Get()
            height, width = size.Get()
        else:
            left, top = topLeft.Get()
            right, bottom = bottomRight.Get()
            width, height = size.Get()

        self.margins = [left, top, right, bottom]
        self.paperSize = wx.Size(height, width)

    def doScale(self, evt):
        try:
            scale = int(self.scaler.GetValue())
        except ValueError:
            self.message.SetLabel('scale must be a number')
            return

        if scale != self.scale:
            self.changed = True

        self._apply_scale(scale)

    def doClose(self, event):
        self.Close()

    def doPageSetup(self, event):
        dialog = wx.PageSetupDialog(self, self.pagedata)
        if dialog.ShowModal() == wx.ID_OK:
            self.pagedata = wx.PageSetupDialogData(dialog.GetPageSetupData())
            self.printdata = wx.PrintData(self.pagedata.GetPrintData())
            self.changed = True
        dialog.Destroy()

    def doPreview(self, event):
        self._read_margins()

        dummy_printout = wx.Printout()  # counteract a bug in wxPython
        dummy_preview = wx.PrintPreview(dummy_printout, None, self.printdata)

        printout = ReportPrintout(self.paperSize, self.margins,
            100.0 / self.scale)
        preview = wx.PrintPreview(printout, None, self.printdata)
        preview.SetZoom(100)

        try:
            self.preview.Close()
        except AttributeError:
            pass

        frame = PreviewFrame(preview, self, "Print Preview")
        frame.Maximize()
        frame.Initialize()
        frame.MakeModal(False) # counteract a bug in wxPython

        bar = frame.GetControlBar()
        bar.SetZoomControl(100)
        sizer = bar.GetSizer()
        for item in sizer.GetChildren():
            item.SetProportion(1)
            item.SetFlag(wx.ALL)
            item.SetBorder(5)
        bar.SetMinSize(sizer.GetMinSize())

        item = frame.GetSizer().GetItem(bar)
        item.SetFlag(wx.GROW | wx.ALL)
        item.SetBorder(5)

        frame.Layout()
        frame.Show(True)
        self.preview = frame

    def doPrint(self, event):
        self._read_margins()
        printout = ReportPrintout(self.paperSize, self.margins,
            100.0 / self.scale)

        self.pdd = wx.PrintDialogData(self.printdata)
        printer = wx.Printer(self.pdd)
        if printer.Print(self, printout, True) == wx.PRINTER_ERROR:
            wx.MessageBox("Error while attempting to print.", "",
                wx.OK | wx.ICON_ERROR)
        else:
            self.pdd = wx.PrintDialogData(printer.GetPrintDialogData())
            self.printdata = wx.PrintData(self.pdd.GetPrintData())

    def onClose(self, event):
        try:
            self._save_data()
        finally:
            event.Skip()

    def onSize(self, event):
        self.Refresh()
        event.Skip()

def Print():
    try:
        Data.print_window.Close()
    except AttributeError:
        pass
    frame = PrintFrame(self)
    ReportAids.RegisterSizePos(frame, "PrintWindow")
    frame.Show(True)
    Data.print_window = frame

Print()
